<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BrandController extends Controller
{
    public function index(Request $request)
    {
        $query = Brand::withCount(['products']);

        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                    ->orWhere('slug', 'LIKE', "%{$search}%");
            });
        }

        $brands = $query->latest()->paginate(100);

        return view('backend.brand.index', compact('brands'));
    }

    public function create()
    {
        return view('backend.brand.create');
    }

    public function edit($id)
    {
        $brand = Brand::findOrFail($id);

        return view('backend.brand.edit', compact('brand'));
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|unique:brands,name',
                'slug' => 'required|string|unique:brands,slug',
                'image' => 'required|image|mimes:jpg,jpeg,png,webp|max:2048',

            ]);

            $slug = $validated['slug'] ?? Str::slug($validated['name']);

            $count = Brand::where('slug', $slug)->count();
            if ($count > 0) {
                $slug .= '-'.time();
            }

            $slug = $validated['slug'] ?? Str::slug($validated['name']);
            if (Brand::where('slug', $slug)->exists()) {
                $slug .= '-'.time();
            }

            $imagePath = null;
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $filename = time().'-'.uniqid().'.'.$image->getClientOriginalExtension();

                $destinationPath = public_path('uploads/brands');

                if (! file_exists($destinationPath)) {
                    mkdir($destinationPath, 0777, true);
                }

                $image->move($destinationPath, $filename);

                $imagePath = 'uploads/brands/'.$filename;
            }

            $brand = Brand::create([
                'name' => $validated['name'],
                'slug' => $slug,
                'image' => $imagePath,
            ]);

            return redirect()->route('admin.brands.index')
                ->with('success', 'Brand created successfully.');

        } catch (\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Something went wrong: '.$e->getMessage())
                ->withInput();
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $brand = Brand::findOrFail($id);

            $validated = $request->validate([
                'name' => 'required|string|unique:brands,name,'.$id,
                'slug' => 'nullable|string|unique:brands,slug,'.$id,
                'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            ]);

            $slug = $validated['slug'] ?? Str::slug($validated['name']);
            $count = Brand::where('slug', $slug)->where('id', '!=', $id)->count();
            if ($count > 0) {
                $slug .= '-'.time();
            }

            $imagePath = $brand->image;

            if ($request->hasFile('image')) {
                if (! empty($brand->image) && file_exists(public_path($brand->image))) {
                    unlink(public_path($brand->image));
                }

                $image = $request->file('image');
                $filename = time().'-'.uniqid().'.'.$image->getClientOriginalExtension();
                $destinationPath = public_path('uploads/brands');

                if (! file_exists($destinationPath)) {
                    mkdir($destinationPath, 0777, true);
                }

                $image->move($destinationPath, $filename);
                $imagePath = 'uploads/brands/'.$filename;
            }

            $brand->update([
                'name' => $validated['name'],
                'slug' => $slug,
                'image' => $imagePath,
            ]);

            return redirect()->route('admin.brands.index')
                ->with('success', 'Brand updated successfully.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->validator)
                ->withInput();
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Something went wrong: '.$e->getMessage())
                ->withInput();
        }
    }

    public function destroy($id)
    {
        try {
            $brand = Brand::findOrFail($id);

            if ($brand->products()->count() > 0) {
                return redirect()->back()->with('error', 'Cannot delete this brand because it has products.');
            }

            if (! empty($brand->image) && file_exists(public_path($brand->image))) {
                unlink(public_path($brand->image));
            }

            $brand->delete();

            return redirect()->back()->with('success', 'Brand deleted successfully.');

        } catch (\ModelNotFoundException $e) {
            return redirect()->back()->with('error', 'Brand not found.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Something went wrong: '.$e->getMessage());
        }
    }
}
