<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Mail\CustomerOtpMail;
use App\Models\Customer;
use App\Models\GmailSmtp;
use App\Models\Order;
use App\Models\ProductReview;
use App\Models\Setting;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CustomerController extends Controller
{
    public function loginForm()
    {
        return view('customer.auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'password' => 'required',
        ]);

        $loginType = filter_var($request->email, FILTER_VALIDATE_EMAIL) ? 'email' : 'phone';

        $credentials = [
            $loginType => $request->email,
            'password' => $request->password,
            'status' => 'active'
        ];

        if (Auth::guard('customer')->attempt($credentials)) {
            return response()->json([
                'success' => true,
                'message' => 'Login successful'
            ], 200);
        }

        $user = Customer::where($loginType, $request->email)->first();

        if ($user && $user->status != 'active') {
            return response()->json([
                'success' => false,
                'message' => 'আপনার অ্যাকাউন্টটি বর্তমানে ইনঅ্যাক্টিভ আছে। দয়া করে এডমিনের সাথে যোগাযোগ করুন।'
            ], 403);
        }

        return response()->json([
            'success' => false,
            'message' => 'ইমেইল/ফোন অথবা পাসওয়ার্ড সঠিক নয়।'
        ], 401);
    }

    public function registerForm()
    {
        return view('customer.auth.register');
    }

    public function registerStore(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:customers,email',
            'phone' => 'required|unique:customers,phone|min:11|max:11',
            'password' => 'required|confirmed',
        ], [
            'email.unique' => 'এই ইমেইলটি ইতিমধ্যে ব্যবহার করা হয়েছে।',
            'phone.unique' => 'এই ফোন নম্বরটি ইতিমধ্যে ব্যবহার করা হয়েছে।',
        ]);

        try {
            $user = Customer::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'password' => Hash::make($request->password),
                'status' => 'active',
            ]);

            Auth::guard('customer')->login($user);

            return response()->json([
                'success' => true,
                'message' => 'Registration successful'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong! Please try again.'
            ], 500);
        }
    }

    public function logout()
    {
        Auth::guard('customer')->logout();
        return redirect()->route('home');
    }

    public function showForgetForm()
    {
        return view('customer.auth.forget');
    }

    public function sendOtp(Request $request)
    {
        $request->validate(['email' => 'required|email|exists:customers,email']);

        $smtp = GmailSmtp::where('status', 1)->first();
        $setting = Setting::first();

        if ($smtp) {
            config([
                'mail.default' => 'smtp',
                'mail.mailers.smtp.transport' => 'smtp',
                'mail.mailers.smtp.host' => $smtp->host,
                'mail.mailers.smtp.port' => $smtp->port,
                'mail.mailers.smtp.username' => $smtp->email,
                'mail.mailers.smtp.password' => $smtp->password,
                'mail.mailers.smtp.encryption' => strtolower($smtp->encryption),
                'mail.from.address' => $smtp->email,
                'mail.from.name' => $setting->name ?? 'Shop',
            ]);
        }

        $customer = Customer::where('email', $request->email)->first();
        $otp = rand(100000, 999999);
        $customer->otp_code = $otp;
        $customer->otp_expires_at = Carbon::now()->addMinutes(10);
        $customer->save();

        Mail::to($customer->email)->send(new CustomerOtpMail($otp));

        return redirect()->route('customer.verify-otp')->with('email', $customer->email);
    }

    public function showOtpForm()
    {
        return view('customer.auth.otp');
    }

    public function verifyOtp(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:customers,email',
            'otp' => 'required|digits:6',
        ]);

        $customer = Customer::where('email', $request->email)->first();

        if ($customer->otp_code != $request->otp || Carbon::now()->gt($customer->otp_expires_at)) {
            return back()->withErrors(['otp' => 'Invalid or expired OTP']);
        }

        return redirect()->route('customer.reset-password')->with('email', $customer->email);
    }

    public function showResetForm()
    {
        return view('customer.auth.reset');
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:customers,email',
            'password' => 'required|confirmed|min:6',
        ]);

        $customer = Customer::where('email', $request->email)->first();
        $customer->password = Hash::make($request->password);
        $customer->otp_code = null;
        $customer->otp_expires_at = null;
        $customer->save();

        return redirect()->route('home')->with('success', 'Password reset successfully!');
    }

    public function dashboard()
    {
        return view('customer.dashboard');
    }

    public function orders()
    {
        $customer = Auth::guard('customer')->user();
        $orders = Order::where('customer_id', $customer->id)
            ->orderBy('created_at', 'desc')
            ->get();

        return view('customer.order', compact('orders'));
    }

    public function show($id)
    {
        $order = Order::with(['items.product', 'shipping'])
            ->where('customer_id', Auth::guard('customer')->id())
            ->findOrFail($id);

        $subtotal = $order->items->sum(function ($item) {
            return $item->price * $item->quantity;
        });

        $total = $subtotal + ($order->shipping_charge ?? 0) - ($order->discount ?? 0);

        return view('customer.order-view', compact('order', 'subtotal', 'total'));
    }

    public function updateAccount(Request $request)
    {
        $user = Auth::guard('customer')->user();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:customers,email,' . $user->id,
            'phone' => 'required|string|max:20',
            'current_password' => 'nullable|required_with:new_password',
            'new_password' => 'nullable|min:6|confirmed',
        ]);

        if ($request->filled('current_password')) {
            if (!Hash::check($request->current_password, $user->password)) {
                return back()->with('error', 'Old password does not match!');
            }
            $user->password = Hash::make($request->new_password);
        }

        $user->name = $request->name;
        $user->email = $request->email;
        $user->address = $request->address;
        $user->phone = $request->phone;
        $user->save();

        return back()->with('success', 'Account updated successfully!');
    }
    
    public function review(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'rating' => 'required|integer|min:1|max:5',
            'title' => 'required|string|max:255',
            'review' => 'required|string',
            'status' => 'required|in:0,1',
        ]);

        DB::beginTransaction();
        try {
            ProductReview::create([
                'product_id' => $request->product_id,
                'rating' => $request->rating,
                'title' => $request->title,
                'review' => $request->review,
                'status' => $request->status,
            ]);

            DB::commit();

            return back()->with('success', 'Review created successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Review Store Error: '.$e->getMessage());

            return back()->withInput()->with('error', 'Failed to create review.');
        }
    }
    
    public function ordersCancel(Request $request, $id)
{
    $order = Order::findOrFail($id);

    if ($order->order_status !== 'pending') {
        return redirect()->back()->with('error', 'Only pending orders can be cancelled.');
    }

    try {
        $order->update([
            'order_status' => 'cancelled'
        ]);

        return redirect()->back()->with('success', 'Order cancelled successfully.');
        
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Something went wrong. Please try again.');
    }
}
}